/*
 * Pseudorandom generators for SWI Prolog
 * Samer Abdallah (2009)
 * Some samplers adapted from code in Tom Minka's lightspeed Matlab toolbox.
 * Other bits of code culled from SWI Prolog distribution random.c
 *
 * To do:
 *    work out lower bound for skew stable distributions
 * 	stream splitting with jumps
 * 	reduce blob copying
 */

#define _USE_MATH_DEFINES 1

#include <SWI-Stream.h>
#include <SWI-Prolog.h>
#include <math.h>
#include <float.h>
#include "RngStream.h"

#if HAVE_GSL
#include <gsl/gsl_sf_zeta.h>
#else
double gsl_sf_zeta(const double s) { return NAN; }
double gsl_sf_hzeta(const double s, const double k) { return NAN; }
#endif

#ifndef M_PI
#define M_PI       3.14159265358979323846
#endif

// random generator state
typedef struct {
	struct rng_state g;
	double prev_normal;
} RndState;

typedef struct {
	struct rng_jump j;
	int e;
} RndJump;

static PL_blob_t rs_blob, rj_blob;
static RndState CurrentState;

double Raw(RndState *);
double Uniform(RndState *);
double Normal(RndState *);
double Exponential(RndState *);
double Gamma(RndState *,double a);
long   Poisson(RndState *,double a);
int    Dirichlet(RndState *,long n, double *a, double *x);
double Beta(RndState *,double a, double b);
double Zeta(RndState *,double a);
int 	 Binomial(RndState *,double p, int n);
double Stable(RndState *,int param,double a, double b);
long   Discrete(RndState *, long n, double *p);

static int randomise_from(FILE *p, RndState *S);
static int spawn_gen(RndState *S0, RndState *S1);

static void InitRndState(RndState *S);

int rs_write(IOSTREAM *,atom_t,int);
int rj_write(IOSTREAM *,atom_t,int);

install_t install();

foreign_t init_rnd_state( term_t s); 
foreign_t get_rnd_state( term_t s); 
foreign_t set_rnd_state( term_t s); 
foreign_t is_rnd_state( term_t s); 
foreign_t randomise( term_t s); 
foreign_t spawn( term_t s0, term_t s1, term_t s2);
foreign_t jump( term_t j, term_t s1, term_t s2);
foreign_t init_jump( term_t n, term_t j);
foreign_t double_jump( term_t j1, term_t j2);

foreign_t sample_Raw( term_t x, term_t s0, term_t s1); 
foreign_t sample_Uniform01( term_t x, term_t s0, term_t s1); 
foreign_t sample_Normal( term_t x, term_t s0, term_t s1); 
foreign_t sample_Exponential( term_t x, term_t s0, term_t s1); 
foreign_t sample_Gamma( term_t a, term_t x, term_t s0, term_t s1);
foreign_t sample_Poisson( term_t a, term_t x, term_t s0, term_t s1);
foreign_t sample_Beta( term_t a, term_t b, term_t x, term_t s0, term_t s1);
foreign_t sample_Zeta( term_t a, term_t x, term_t s0, term_t s1);
foreign_t sample_Dirichlet( term_t n, term_t a, term_t x, term_t s0, term_t s1);
foreign_t sample_Binomial( term_t p, term_t n, term_t x, term_t s0, term_t s1);
foreign_t sample_Stable( term_t a, term_t b, term_t x, term_t s0, term_t s1);
foreign_t sample_Dirichlet( term_t n, term_t a, term_t x, term_t s0, term_t s1);
foreign_t sample_Discrete( term_t n, term_t p, term_t x, term_t s0, term_t s1);

foreign_t sample_Single_( term_t x); 
foreign_t sample_Double_( term_t x); 

static void test() 
{
	RndState S;

	clock_t t1,t2;
	int	i, N=1000000;
	double dt;

	InitRndState(&S);

	t1=clock();
	for (i=0; i<N; i++) Uniform(&S);
	t2=clock();
	dt=(t2-t1)/(double)CLOCKS_PER_SEC;
	printf("%d doubles in %lf s --> %lf k doubles/s\n",N,dt,N/(1000.0*dt));
}

install_t install() { 
	PL_register_foreign("init_rnd_state", 1, (void *)get_rnd_state, 0);
	PL_register_foreign("get_rnd_state", 1, (void *)get_rnd_state, 0);
	PL_register_foreign("set_rnd_state", 1, (void *)set_rnd_state, 0);
	PL_register_foreign("is_rnd_state", 1, (void *)is_rnd_state, 0);
	PL_register_foreign("randomise", 1, (void *)randomise, 0);
	PL_register_foreign("spawn", 3, (void *)spawn, 0);
	PL_register_foreign("jump", 3, (void *)jump, 0);

	PL_register_foreign("init_jump", 2, (void *)init_jump, 0);
	PL_register_foreign("double_jump", 2, (void *)double_jump, 0);

	PL_register_foreign("sample_Raw", 3, (void *)sample_Raw, 0);

	PL_register_foreign("sample_Uniform01", 3, (void *)sample_Uniform01, 0);
	PL_register_foreign("sample_Normal",    3, (void *)sample_Normal, 0);
	PL_register_foreign("sample_Exponential",    3, (void *)sample_Exponential, 0);
	PL_register_foreign("sample_Gamma",     4, (void *)sample_Gamma, 0);
	PL_register_foreign("sample_Poisson",   4, (void *)sample_Poisson, 0);
	PL_register_foreign("sample_Beta",      5, (void *)sample_Beta, 0);
	PL_register_foreign("sample_Zeta",     4, (void *)sample_Zeta, 0);
	PL_register_foreign("sample_Dirichlet", 5, (void *)sample_Dirichlet, 0);
	PL_register_foreign("sample_Binomial",  5, (void *)sample_Binomial, 0);
	PL_register_foreign("sample_Stable",      5, (void *)sample_Stable, 0);
	PL_register_foreign("sample_Discrete",  5, (void *)sample_Discrete, 0);

	PL_register_foreign("sample_Single_", 1, (void *)sample_Single_, 0);
	PL_register_foreign("sample_Double_", 1, (void *)sample_Double_, 0);

	rs_blob.magic = PL_BLOB_MAGIC;
	rs_blob.flags = 0; // PL_BLOB_UNIQUE;
	rs_blob.name = "rndstate";
	rs_blob.acquire = 0; // rs_acquire;
	rs_blob.release = 0; // rs_release;
	rs_blob.compare = 0; // rs_compare;
	rs_blob.write   = rs_write;

	rj_blob.magic = PL_BLOB_MAGIC;
	rj_blob.flags = 0; // PL_BLOB_UNIQUE;
	rj_blob.name = "rndjump";
	rj_blob.acquire = 0; // rs_acquire;
	rj_blob.release = 0; // rs_release;
	rj_blob.compare = 0; // rs_compare;
	rj_blob.write   = rj_write;

	InitRndState(&CurrentState);
	test();
}


// throws a Prolog exception to signal type error
static int type_error(term_t actual, const char *expected)
{ 
	term_t ex = PL_new_term_ref();

  PL_unify_term(ex, PL_FUNCTOR_CHARS, "error", 2,
		      PL_FUNCTOR_CHARS, "type_error", 2,
		        PL_CHARS, expected,
		        PL_TERM, actual,
		      PL_VARIABLE);

  return PL_raise_exception(ex);
}

static int memory_error(size_t amount)
{ 
	term_t ex = PL_new_term_ref();

  PL_unify_term(ex, PL_FUNCTOR_CHARS, "error", 2,
		      PL_FUNCTOR_CHARS, "memory_error", 1,
		        PL_INTEGER, amount,
		      PL_VARIABLE);

  return PL_raise_exception(ex);
}

// extract double from Prolog float
static int get_double(term_t term, double *p)
{ 
	if (PL_get_float(term, p)) return TRUE; 
	else return type_error(term, "float");
}

// extract long from Prolog integer
static int get_long(term_t term, long *p)
{ 
	if (PL_get_long(term, p)) return TRUE; 
	else return type_error(term, "integer");
}


// unify Prolog list of floats with array of doubles
static int unify_list_doubles(term_t list, double *x, int n)
{
	int i;
	list=PL_copy_term_ref(list);

	for (i=0; i<n; i++) {
		term_t head=PL_new_term_ref();
		term_t tail=PL_new_term_ref();
		if (!PL_unify_list(list,head,tail)) PL_fail; 
		if (!PL_unify_float(head,x[i])) PL_fail;
		list=tail;
	}
	return PL_unify_nil(list);
}

// read list of floats from term and write to double array
static int get_list_doubles(term_t list, double *vals)
{
	term_t  head=PL_new_term_ref();
	int 		n;

	// copy term ref so as not to modify original
	list=PL_copy_term_ref(list);
	for (n=0;PL_get_list(list,head,list);n++) {
			if (!PL_get_float(head,&vals[n])) return FALSE;
	}
	if (!PL_get_nil(list)) return FALSE; 
	return TRUE;
}


// unify Prolog BLOB with RndState structure
static int unify_state(term_t state,RndState *S) {
	return PL_unify_blob(state, S, sizeof(RndState), &rs_blob); 
}

// extract RndState structure from Prolog BLOB
static int get_state(term_t state, RndState *S0)
{ 
	PL_blob_t *type;
	size_t    len;
	RndState *S;
  
	PL_get_blob(state, (void **)&S, &len, &type);
	if (type != &rs_blob) {
		return type_error(state, "rndstate");
	} else {
		*S0=*S;
		return TRUE;
	}
} 

int rs_write(IOSTREAM *s, atom_t a, int flags) 
{ 
	PL_blob_t *type;
	size_t    len;
	RndState *p=(RndState *)PL_blob_data(a,&len,&type);
	if (p) {
		int i;
		Sfprintf(s,"<rs");
		for (i=0; i<6; i++) Sfprintf(s," %.0lf",p->g.Cg[i]);
		Sfprintf(s,">");
	}
	return TRUE; 
}

// unify Prolog BLOB with RndJump structure
static int unify_jump(term_t jump,RndJump *J) {
	return PL_unify_blob(jump, J, sizeof(RndJump), &rj_blob); 
}

// extract RndState structure from Prolog BLOB
static int get_jump(term_t jump, RndJump *J0)
{ 
	PL_blob_t *type;
	size_t    len;
	RndJump *J;
  
	PL_get_blob(jump, (void **)&J, &len, &type);
	if (type != &rj_blob) {
		return type_error(jump, "rndjump");
	} else {
		*J0=*J;
		return TRUE;
	}
} 

int rj_write(IOSTREAM *s, atom_t a, int flags) 
{ 
	PL_blob_t *type;
	size_t    len;
	RndJump *p=(RndJump *)PL_blob_data(a,&len,&type);
	if (p) {
		Sfprintf(s,"<rj %d>",p->e);
	}
	return TRUE; 
}

static int alloc_array(size_t N, size_t SZ, void **PP) {
	*PP=calloc(N,SZ);
	if (*PP) return TRUE;
	else return memory_error(N*SZ);
}

// -----------------------------------------------------

// unify jump with a new blob representing a sequence jump
foreign_t init_jump(term_t pow, term_t jump) { 
	RndJump J;
	long e;
	
	if (get_long(pow,&e)) {
		J.e=e;
		RngStream_InitJump(&J.j,e);
		return unify_jump(jump,&J);
	} else return FALSE;
}

foreign_t double_jump(term_t j0, term_t j1) { 
	RndJump J;
	
	if (get_jump(j0,&J)) {
		J.e++;
		RngStream_DoubleJump(&J.j,&J.j);
		return unify_jump(j1,&J);
	} else return FALSE;
}

// unify Prolog term with the default random state 
foreign_t init_rnd_state(term_t state) { 
	RndState S0;
	InitRndState(&S0);
	return unify_state(state,&S0);
}

// unify Prolog term with current random state structure
foreign_t get_rnd_state(term_t state) { return unify_state(state,&CurrentState); }

// set current random state structure to values in Prolog term
foreign_t set_rnd_state(term_t state) { return get_state(state,&CurrentState); }

// get truly random state from /dev/random
foreign_t randomise(term_t state) { 
	RndState S;
	FILE		*p;

	p=fopen("/dev/random","r");
	if (p!=NULL) {
		randomise_from(p,&S);
		fclose(p);
		return unify_state(state,&S); 
	}
	{
		term_t ex = PL_new_term_ref();

	  PL_unify_term(ex, PL_FUNCTOR_CHARS, "error", 2,
					PL_FUNCTOR_CHARS, "file_error", 1,
					  PL_CHARS, "fread /dev/random",
					PL_VARIABLE);

	  return PL_raise_exception(ex);
	}
}
	
foreign_t spawn(term_t new, term_t orig, term_t next) { 
	RndState S0, S1;

	return get_state(orig,&S0)
			&& spawn_gen(&S0,&S1)
			&& unify_state(next,&S0) 
			&& unify_state(new,&S1); 
}

foreign_t jump(term_t j, term_t s0, term_t s1) { 
	RndState S;
	RndJump  J;

	if (get_jump(j,&J) && get_state(s0,&S)) {
		RngStream_Advance(&J.j,&S.g,&S.g);
		return unify_state(s1,&S); 
	} else return FALSE;
}

// set current random state structure to values in Prolog term
foreign_t is_rnd_state(term_t state) { 
	PL_blob_t *type;
	return PL_is_blob(state,&type) && type==&rs_blob;
}

// unify one term with numeric value and other with random state structure
static int unify_float_state(term_t x, double z, term_t s, RndState *S) {
	return  PL_unify_float(x, z) && unify_state(s,S);
}
	
// unify one term with numeric value and other with random state structure
static int unify_integer(term_t x, long z, term_t s, RndState *S) {
	return  PL_unify_integer(x, z) && unify_state(s,S);
}
	
foreign_t sample_Raw(term_t x, term_t s0, term_t s1) {
	RndState S;
	return get_state(s0,&S) &&
	       unify_float_state(x, Raw(&S), s1, &S);
}


foreign_t sample_Uniform01(term_t x, term_t s0, term_t s1) {
	RndState S;
	return get_state(s0,&S) &&
	       unify_float_state(x, Uniform(&S), s1, &S);
}


foreign_t sample_Normal(term_t x, term_t s0, term_t s1) {
	RndState S;
	return get_state(s0,&S) &&
		    unify_float_state(x, Normal(&S), s1, &S);
}

foreign_t sample_Exponential(term_t x, term_t s0, term_t s1) {
	RndState S;
	return get_state(s0,&S) &&
		    unify_float_state(x, Exponential(&S), s1, &S);
}



foreign_t sample_Gamma(term_t a, term_t x, term_t s0, term_t s1) {
	RndState S;
	double A;

	return get_state(s0,&S) &&
		    get_double(a,&A) &&
			 unify_float_state(x, Gamma(&S,A), s1, &S);
}

foreign_t sample_Poisson(term_t a, term_t x, term_t s0, term_t s1) {
	RndState S;
	double A;

	return get_state(s0,&S) &&
		    get_double(a,&A) &&
	       PL_unify_integer(x, Poisson(&S,A)) &&
			 unify_state(s1,&S);
}


foreign_t sample_Dirichlet(term_t n, term_t a, term_t x, term_t s0, term_t s1)
{
	long N;

	if (get_long(n,&N)) {
		double   *A, *X;
		RndState S;

		int rc = alloc_array(N,sizeof(double),(void **)&A) 
				&& alloc_array(N,sizeof(double),(void **)&X) 
				&& get_list_doubles(a,A)
				&& get_state(s0,&S)
				&& Dirichlet(&S,N,A,X)
				&& unify_list_doubles(x,X,N)
				&& unify_state(s1,&S);

		if (X) free(X);
		if (A) free(A);
		return rc;
	} else return FALSE;
}



foreign_t sample_Beta(term_t a, term_t b, term_t x, term_t s0, term_t s1) {
	RndState S;
	double A, B;

	return get_state(s0,&S) &&
		    get_double(a,&A) &&
	       get_double(b,&B) && 
			 unify_float_state(x, Beta(&S,A,B),s1,&S);
}

foreign_t sample_Zeta(term_t a, term_t x, term_t s0, term_t s1) {
	RndState S;
	double A;

	return get_state(s0,&S) &&
		    get_double(a,&A) &&
			 PL_unify_float(x, Zeta(&CurrentState,A)) &&
			 unify_state(s1,&S);
}



foreign_t sample_Binomial(term_t p, term_t n, term_t x, term_t s0, term_t s1) {
	RndState S;
	double P;
	long   N;

	return get_state(s0,&S) &&
		    get_double(p,&P) &&
	       get_long(n,&N) && 
	       PL_unify_integer(x, Binomial(&S,P,(int)N)) &&
			 unify_state(s1,&S);
}


foreign_t sample_Stable(term_t a, term_t b, term_t x, term_t s0, term_t s1) {
	RndState S;
	double A, B;

	return get_state(s0,&S) &&
		    get_double(a,&A) &&
	       get_double(b,&B) && 
			 unify_float_state(x, Stable(&S,1,A,B),s1,&S);
}

foreign_t sample_Discrete(term_t n, term_t p, term_t x, term_t s0, term_t s1)
{
	long N;

	if (get_long(n,&N)) {
		double   *P;
		RndState S;

		int rc = alloc_array(N,sizeof(double),(void **)&P) 
				&& get_list_doubles(p,P)
				&& get_state(s0,&S)
				&& PL_unify_integer(x, 1+Discrete(&S,N,P));
			   unify_state(s1,&S);

		if (P) free(P);
		return rc;
	} else return FALSE;
}

// Returns a sample from Normal(0,1)
double Normal(RndState *S)
{
	double x=S->prev_normal;

	if(!isnan(x)) {
		S->prev_normal=NAN;
	} else {
		double y,radius;

		/* Generate a random point inside the unit circle */
		do {
			x = 2*Uniform(S)-1;
			y = 2*Uniform(S)-1;
			radius = (x*x)+(y*y);
		} while((radius >= 1.0) || (radius == 0.0));
		/* Box-Muller formula */
		radius = sqrt(-2*log(radius)/radius);
		x *= radius;
		y *= radius;
		S->prev_normal=y;
	}
	return x;
}

/* Returns a sample from Gamma(a, 1).
 * For Gamma(a,b), scale the result by b.
 */
double Gamma(RndState *S, double a)
{
  /* Algorithm:
   * G. Marsaglia and W.W. Tsang, A simple method for generating gamma
   * variables, ACM Transactions on Mathematical Software, Vol. 26, No. 3,
   * Pages 363-372, September, 2000.
   * http://portal.acm.org/citation.cfm?id=358414
   */
  double boost, d, c, v;
  if(a < 1) {
    /* boost using Marsaglia's (1961) method: gam(a) = gam(a+1)*U^(1/a) */
    boost = exp(log(Uniform(S))/a);
    a++;
  } 
  else boost = 1;
  d = a-1.0/3; c = 1.0/sqrt(9*d);
  while(1) {
    double x,u;
    do {
      x = Normal(S);
      v = 1+c*x;
    } while(v <= 0);
    v = v*v*v;
    x = x*x;
    u = Uniform(S);
    if((u < 1-.0331*x*x) || 
       (log(u) < 0.5*x + d*(1-v+log(v)))) break;
  }
  return( boost*d*v );
}

/* Returns a sample from Dirichlet(n,a) where n is dimensionality */
int Dirichlet(RndState *S, long n, double *a, double *x)
{
	long i;
	double tot=0, z;
	for (i=0; i<n; i++) { z=Gamma(S,a[i]); tot+=z; x[i]=z; }
	for (i=0; i<n; i++) { x[i]/=tot; }
	return TRUE;
}

/* Returns a sample from Beta(a,b) */
double Beta(RndState *S, double a, double b)
{
  double g = Gamma(S,a);
  return g/(g + Gamma(S,b));
}

double Hyperbolic(RndState *S, double a)
{
	return floor(pow(1-Uniform(S),-1/a));
}

/* Sample from zeta distribution
 *
 * This is an inverse CDF method. It works by using  an 
 * approximation of the Hurwitz Zeta function and then
 * walking left or right until we get to the correct point.
 * The approximation is good for large values so usually we
 * only have to walk left or right a few steps for small 
 * values.
 */
double Zeta(RndState *S, double s)
{
	double v=(1-Uniform(S))*gsl_sf_zeta(s);
	double k0, k=round(pow(v*(s-1),1/(1-s))); // approx inverse of hzeta
	double zk0, zk=gsl_sf_hzeta(s,k);

	if (zk>=v) {
		do {
			k0=k; zk0=zk;
			k=k0+1; zk=zk0-pow(k0,-s);
		} while (zk>=v && zk!=zk0);
		return k0;
	} else {
		do {
			k0=k; zk0=zk;
			k=k0-1; zk=zk0+pow(k,-s);
		} while (zk<v && zk!=zk0);
		return k;
	}
}

/* Very fast binomial sampler. 
 * Returns the number of successes out of n trials, with success probability p.
 */
int Binomial(RndState *S, double p, int n)
{
  int r = 0;
  if(isnan(p)) return 0;
  if(p < DBL_EPSILON) return 0;
  if(p >= 1-DBL_EPSILON) return n;
  if((p > 0.5) && (n < 15)) {
    /* Coin flip method. This takes O(n) time. */
    int i;
    for(i=0;i<n;i++) {
      if(Uniform(S) < p) r++;
    }
    return r;
  }
  if(n*p < 10) {
    /* Waiting time method.  This takes O(np) time. */
    double q = -log(1-p), e = -log(Uniform(S)), s;
    r = n;
    for(s = e/r; s <= q; s += e/r) {
      r--;
      if(r == 0) break;
      e = -log(Uniform(S));
    }
    r = n-r;
    return r;
  }
  if (1) {
    /* Recursive method.  This makes O(log(log(n))) recursive calls. */
    int i = (int)floor(p*(n+1));
    double b = Beta(S,i, n+1-i);
    if(b <= p) r = i + Binomial(S,(p-b)/(1-b), n-i);
    else r = i - 1 - Binomial(S,(b-p)/b, i-1);
    return r;
  }
}

double Exponential(RndState *S) { return -log(1-Uniform(S)); }

long Poisson(RndState *S, double lambda)
{
	long r;
	if (lambda>=15) {
		double m=floor(lambda*7/8);
		double x=Gamma(S,m);

		if (x>lambda) r=Binomial(S,lambda/x,m-1);
		else          r=m+Poisson(S,lambda-x);
	} else {
		double p;
		for (p=0, r=-1; p<lambda; p+=Exponential(S)) r++; 
	}
	return r;
}

/* Returns a sample from stable distribution */
double Stable(RndState *S, int param, double alpha, double beta)
{
	double X, theta=M_PI*(Uniform(S)-0.5);

	// These methods come from John Nolan's book about
	// stable distributions. They return standardised stable random 
	// numbers in the S(alpha,beta;0) parameterisation

	if (beta==0) {
		if (alpha==1) {
			X=tan(theta);
		} else {
			double W=Exponential(S);
			X=sin(alpha*theta)/cos(theta)*pow(cos((alpha-1)*theta)/(W*cos(theta)),1/alpha-1);
		}
	} else {
		double W=Exponential(S);
		double zeta=beta*tan(alpha*M_PI/2);
		if (alpha==1) {
			double b=2*beta/M_PI;
			double bt=1+b*theta;
			X=bt*tan(theta) - b*log(W*cos(theta)/bt);
		} else {
			double ath = alpha*theta; 
			double a1th = theta-ath; 
			X = ((sin(ath) + zeta*cos(ath))/cos(theta))
				 * pow((cos(a1th) + zeta*sin(a1th))/(W*cos(theta)),1/alpha-1);
			if (param==0) X=X-zeta;
		}
	}
	return X;
}

/* Returns a sample from Dirichlet(n,a) where n is dimensionality */
long Discrete(RndState *S, long n, double *p)
{
	double tot, u;
	long i;

	for (i=0, tot=0; i<n; tot+=p[i++]);
	for (i=0, u=tot*Uniform(S)-p[0]; u>=0; u-=p[++i]); 
	return i;
}

double Uniform(RndState *S0)
{
	return RngStream_Double(&S0->g,&S0->g);
}

double Raw(RndState *S0)
{
	return RngStream_Raw(&S0->g,&S0->g);
}

foreign_t sample_Single_(term_t x) {
	return PL_unify_float(x, RngStream_Float(&CurrentState.g,&CurrentState.g));
}

foreign_t sample_Double_(term_t x) {
	return PL_unify_float(x, RngStream_Double(&CurrentState.g,&CurrentState.g));
}

void InitRndState(RndState *S)
{
	RngStream_InitState(&S->g);
	S->prev_normal=NAN;
}


int spawn_gen(RndState *S0, RndState *S1)
{
	unsigned long seeds[6];
	int i;

	for (i=0; i<6; i++) seeds[i]=(unsigned long)RngStream_Raw(&S0->g,&S0->g);
	RngStream_SetState(&S1->g,seeds);
	for (i=0; i<3; i++) RngStream_Raw(&S1->g,&S1->g);
	S1->prev_normal=NAN;
	return TRUE;
}

int randomise_from(FILE *p, RndState *S)
{
	unsigned long seeds[6];
	size_t r=fread((void *)seeds,sizeof(unsigned long),6,p);

	if (r==6) {
		int i;
		RngStream_SetState(&S->g,seeds);
		for (i=0; i<3; i++) RngStream_Raw(&S->g,&S->g);
		S->prev_normal=NAN;
		return TRUE;
	} else return FALSE;
}

