/*
 * Prolog part of random generator library
 * Samer Abdallah (2009)
*/
	  
:- module(plmidi, [
		midi_mk_outlet/2		% +N:positive_integer, -Ref
	,	midi_is_outlet/1		% +Ref
	,	midi_send/4		% +Ref, +Msg:between(0,255), +Arg1:between(0,127), +Arg2:between(0,127)
	,	midi_send/5		% +Ref, +Msg:between(0,255), +Arg1:between(0,127), +Arg2:between(0,127), +Time:float
	,	midi/3     		% +Ref, +Time:float, +Event
	,	midi_listall/0	
	,	midi_calibrate/0
	]).
	
/** <module> MIDI event output

This module provides the ability to send MIDI events. It uses the
Mac OS X CoreMIDI framework. Events can be sent for immediate
dispatch or scheduled for future dispatch at a given time.
Times are specified in seconds since 1st Jan 1970, ie as returned
by get_time/1.

@author Samer Abdallah
*/

:-	load_foreign_library(foreign(plmidi)).

%% midi_listall is det.
%
%  Prints a list of all the available MIDI destinations on the system

%% midi_mk_outlet( +N:positive_integer, -Ref) is semidet.
%
%  Opens a connection to the Nth CoreMIDI destination. Fails
%  if there are fewer than N destinations available. A list of
%  available destinations can be printed to stdout using 
%  midi_listall/0. Resources associated with the connection
%  will be released when the Ref atom is reclaimed by the garbage
%  collector.
%
%  @param N is the index of the destination to open, 1 is the first.
%  @param Ref is a BLOB atom representing the connection.
%
%  @see midi_listall/0

%% midi_is_outlet(+Ref) is semidet.
%
%  Determines whether or not Ref is a MIDI connection BLOB as returned
%  by midi_mk_outlet/2.
%
%  @see midi_mk_outlet/2

%% midi_send(+Ref, +Msg:between(0,255), +Arg1:between(0,127) +Arg2:between(0,127)) is det.
%% midi_send(+Ref, +Msg:between(0,255), +Arg1:between(0,127) +Arg2:between(0,127), +Time:float) is det.
%
%  Send a MIDI message to an established destination, with a timestamp
%  of <now> (midi_send/4) or a given time (midi_send/5).
%  or at the specified time. 
%
%  @param Ref is an atom as returned by midi_mk_outlet/2.
%  @param Time is a Unix time as returnd by get_time/1.
%
%  @see midi_mk_outlet/2.
%  @see get_time/1.
midi_send(O,M,A1,A2) :- M1 is M, midi_send_now(O,M1,A1,A2).
midi_send(O,M,A1,A2,T) :- T1 is T, M1 is M, midi_send_at(O,M1,A1,A2,T1).



%% midi(+Ref, +Time:float, +Event) is semidet.
%
%  Schedule a MIDI event, possibly consisting of multiple messages, determined
%  by the Event term.
%
%  @param Ref is an atom as returned by midi_mk_outlet/2.
%  @param Time is a Unix time as returnd by get_time/1.
%  @param Event is a term describing an event, and is one of:
%  
%  * noteon(+Chan:between(0,15),+NoteNum:between(0,127),+Vel:between(0,127))
%  A note-on event with the given channel, pitch (NoteNum) and loudness (Vel).
%  * noteoff(+Chan:between(0,15),+NoteNum:between(0,127))
%  A note-off event with the given channel and pitch (NoteNum).
%  * note(+Chan:between(0,15),+NoteNum:between(0,127),+Vel:between(0,127),+Dur:float)
%  A complete note event (on and off) with the given duration.
%  * prog(+Chan:between(0,15),+Prog:between(0,127))
%  Programme change event in the given channel.
%  * prog(+Chan:between(0,15),+Prog:between(0,127),+Bank:between(0,16383))
%  Combined programme and bank change event in the given channel.
%
%  @see midi_mk_outlet/2.
%  @see get_time/1.

midi(O,T,noteon(Ch,NN,V)) :- midi_send(O,144+Ch,NN,V,T).
midi(O,T,noteoff(Ch,NN)) :- midi_send(O,128+Ch,NN,0,T).

midi(O,T,note(Ch,NN,Vel,Dur)) :- 
	N1 is NN, V1 is Vel,
	midi_send(O,144+Ch,N1,V1,T),
	midi_send(O,128+Ch,N1,0,T+Dur).

midi(O,T,prog(Ch,Prog)) :-
	midi_send(O,192+Ch,Prog,Prog,T).
	
midi(O,T,prog(Ch,Prog,Bank)) :-
	MSB is Bank // 128,
	LSB is Bank mod 128,
	midi_send(O,176+Ch,0,MSB,T),
	midi_send(O,176+Ch,32,LSB,T),
	midi(O,T,prog(Ch,Prog)).


%% midi_calibrate is det.
%
%  Establish a calibrated relationship between Unix time (seconds since
%  1st Jan 1970) and the timebase used by the CoreMIDI framework, which is
%  mach system time and is counted in bus cycles since system boot. The
%  relationship is not trivial since mach system time does not advance
%  while the system is asleep (eg a laptop is closed). When the module
%  is loaded, the calibration is done, and an IOKit wake/sleep event handler 
%  is installed. This should recalibrate the timer whenever the system
%  is awoken from sleep, but midi_calibrate/0 can be called at any time
%  just incase. 

