/*
 *  plmidi - MIDI output for SWI Prolog on Mac OS X
 *
 *  Copyright (C) 2009 Samer Abdallah
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */


#include <SWI-Stream.h>
#include <SWI-Prolog.h>

#include <mach/mach_time.h>
#include <mach/mach_port.h>
#include <mach/mach_interface.h>
#include <mach/mach_init.h>
#include <CoreMIDI/MIDIServices.h>
#include <IOKit/pwr_mgt/IOPMLib.h>
#include <IOKit/IOMessage.h>


#include <stdio.h>
#include <string.h>
#include <pthread.h>
#include <sys/time.h>

// data pertaining to a CoreMIDI connection
typedef struct midi_outlet {
	MIDIClientRef		client;
	MIDIPortRef	  		port;
	MIDIEndpointRef	dest;
} midi_outlet_t;

// for representing a CoreMIDI connection as an atom
static PL_blob_t outlet_blob;

// for mapping Unix time to CoreMIDI timestamps
static double timebase_origin=0;
static double timebase_freq=0;
static mach_timebase_info_data_t timebase;

// for wake/sleep notification
static io_connect_t  root_port; 


// --------------------------   Time conversion


// convert gettimeofday() timeval to microseconds since 1970
static uint64_t timeval_to_micros(struct timeval *tv) {
	return (uint64_t)tv->tv_sec*1000000 + tv->tv_usec;
}

// convert mach system time to microseconds 
static double machtime_to_micros(uint64_t abstime) {
	return abstime*((double)timebase.numer/(double)timebase.denom)/1000.0;
}

// establish relationship between Unix time and mach system time
static void calibrate_timer(int verbose) 
{
	uint64_t mt0, mt1, mt2, tc1;
	double   tm0, tm1, tm2;
	double   off1, off2, moff;
	struct  timeval ct1;

	printf("\n-- plmidi: calibrating MIDI timebase...\n");
	mach_timebase_info(&timebase); // gets conversion factor as a rational number
	// printf("mach timebase: %u/%u\n",timebase.numer,timebase.denom);

	mt0=mach_absolute_time();
	gettimeofday(&ct1,NULL);
	mt1=mach_absolute_time();
	mt2=mach_absolute_time();

	tm0=machtime_to_micros(mt0);
	tm1=machtime_to_micros(mt1);
	tm2=machtime_to_micros(mt2); 
	tc1=timeval_to_micros(&ct1);

	if (verbose) {
		printf("mach_absolute_time() duration = %llu ticks = %lf us\n",mt2-mt1,tm2-tm1);
		printf("gettimeofday()       duration = %llu ticks = %lf us\n",(mt1-mt0)-(mt2-mt1),(tm1-tm0)-(tm2-tm1));
		printf("\n");

		printf("mach time = %lf us\n",tm1);
		printf("unix time = %lld us\n",tc1);
		printf("\n");
	}

	// compute difference between Unix time and mach time in 
	// two ways using measurements taken above, then take mean
	off1=tc1-tm0; off2=tc1-tm1; 
	moff=(off1+off2)/2; 

	timebase_origin = moff/1000000;                    
	timebase_freq = 1e9*timebase.denom/timebase.numer; 

	if (verbose) {
		printf("timebase origin = %lf s\n",timebase_origin);
		printf("timebase freq   = %lf Hz\n",timebase_freq);
	}
	printf("-- plmidi: done calibrating MIDI timebase.\n\n");
}
	


// --------------------------   Wake notification


void sleep_cb(void *refCon, io_service_t service, natural_t msg_type, void * msg_arg )
{
    switch (msg_type) {
        case kIOMessageCanSystemSleep:
        case kIOMessageSystemWillSleep:
			  	printf("\n-- system about to sleep\n");
            IOAllowPowerChange( root_port, (long)msg_arg);
            break;

        case kIOMessageSystemHasPoweredOn: 
			  	printf("-- system has woken up\n");
				calibrate_timer(0); 
				break;

        default: break;
    }
}


void *install_sleep_handler(void *dummy)
{
    IONotificationPortRef  notifyPortRef; // notification port allocated by IORegisterForSystemPower
    io_object_t            notifierObject; // notifier object, used to deregister later
    void*                  refCon; // this parameter is passed to the callback

    // register to receive system sleep notifications

	 printf("\n-- registering plmidi wake/sleep handler...\n");
    root_port = IORegisterForSystemPower( refCon, &notifyPortRef, sleep_cb, &notifierObject );
    if (root_port==0) {
        printf("*** IORegisterForSystemPower failed\n");
        printf("*** You must call midi_calibrate manually after system sleep\n");
		  return NULL;
    }

    // add the notification port to the application runloop
    CFRunLoopAddSource( CFRunLoopGetCurrent(),
            IONotificationPortGetRunLoopSource(notifyPortRef), kCFRunLoopCommonModes ); 

	 printf("-- wake/sleep handler thread running...\n\n");
    CFRunLoopRun();
	 printf("-- wake/sleep handler thread terminating.\n");
    return NULL;
}



// --------------------------   MIDI API


// establish a connection to a MIDI destination and fill midi_outlet_t
// structure with relevant handles.
static int open_outlet(int id, midi_outlet_t *outlet)
{
	outlet->client = NULL;
	outlet->dest = NULL;
	outlet->port = NULL;

	MIDIClientCreate(CFSTR("plmidi"), NULL, NULL, &outlet->client);

	if (outlet->client!=NULL) {
		MIDIOutputPortCreate(outlet->client, CFSTR("plmidi_out"), &outlet->port);
	
		if (outlet->port!=NULL) {
			if (id<MIDIGetNumberOfDestinations()) outlet->dest=MIDIGetDestination(id);
			if (outlet->dest!=NULL) {
				CFStringRef pname;
				char name[64];

				MIDIObjectGetStringProperty(outlet->dest, kMIDIPropertyName, &pname);
				CFStringGetCString(pname, name, sizeof(name), 0);
				CFRelease(pname);
				printf("MIDI outlet opened: %s\n", name);
				return TRUE;
			} else {
				MIDIPortDispose(outlet->port);
				return FALSE;
			}
		} else {
			MIDIClientDispose(outlet->client);
			return FALSE;
		}
	} else {
		return FALSE;
	}
}

static int close_outlet(midi_outlet_t *outlet) 
{
	printf("MIDI outlet closing...\n");
	outlet->dest=NULL;
	if (outlet->port!=NULL) MIDIPortDispose(outlet->port);
	if (outlet->client!=NULL) MIDIClientDispose(outlet->client);
	return TRUE;
}
	

// List all available MIDI destinations to stdout
static void list_devices()
{
	int i, n;
	CFStringRef pname, pmanuf, pmodel;
	char name[64], manuf[64], model[64];

	n = MIDIGetNumberOfDevices();
	for (i = 0; i < n; ++i) {
		MIDIDeviceRef dev = MIDIGetDevice(i);
		
		MIDIObjectGetStringProperty(dev, kMIDIPropertyName, &pname);
		MIDIObjectGetStringProperty(dev, kMIDIPropertyManufacturer, &pmanuf);
		MIDIObjectGetStringProperty(dev, kMIDIPropertyModel, &pmodel);
		
		CFStringGetCString(pname, name, sizeof(name), 0);
		CFStringGetCString(pmanuf, manuf, sizeof(manuf), 0);
		CFStringGetCString(pmodel, model, sizeof(model), 0);
		CFRelease(pname);
		CFRelease(pmanuf);
		CFRelease(pmodel);

		printf("name=%s, manuf=%s, model=%s\n", name, manuf, model);
	}
}


// send a MIDI message
static int send_msg(
		midi_outlet_t *outlet, MIDITimeStamp ts, 
		unsigned char msg, unsigned char arg1, unsigned char arg2)
{
	MIDIPacketList pktlist;
	MIDIPacket     *pkt = &pktlist.packet[0];

	pktlist.numPackets = 1;
	pkt->timeStamp = ts;
	pkt->length = 3;
	pkt->data[0] = msg;
	pkt->data[1] = arg1;
	pkt->data[2] = arg2;

	MIDISend(outlet->port, outlet->dest, &pktlist);
	return TRUE;
}



// --------------------------   Prolog boilerplate

install_t install();

foreign_t listall(); 
foreign_t calibrate();
foreign_t mk_outlet( term_t id, term_t outlet); 
foreign_t is_outlet( term_t outlet); 
foreign_t send_now( term_t addr, term_t msg, term_t arg1, term_t arg2); 
foreign_t send_at( term_t addr, term_t msg, term_t arg1, term_t arg2, term_t time); 

int outlet_release(atom_t a)
{
	PL_blob_t *type;
	size_t    len;
	void *p;

	p=PL_blob_data(a,&len,&type);
	if (p) {
		close_outlet((midi_outlet_t *)p);
	}
	return TRUE;
}

install_t install() 
{ 
	PL_register_foreign("midi_mk_outlet",  2, (void *)mk_outlet, 0);
	PL_register_foreign("midi_is_outlet",  1, (void *)is_outlet, 0);
	PL_register_foreign("midi_send_now",   4, (void *)send_now, 0);
	PL_register_foreign("midi_send_at",    5, (void *)send_at, 0);
	PL_register_foreign("midi_listall",    0, (void *)listall, 0);
	PL_register_foreign("midi_calibrate",  0, (void *)calibrate, 0);

	// F_rand4 = PL_new_functor(PL_new_atom("rand"), 4);

	outlet_blob.magic = PL_BLOB_MAGIC;
	outlet_blob.flags = PL_BLOB_UNIQUE;
	outlet_blob.name = "plmidi_outlet";
	outlet_blob.acquire = 0; // rs_acquire;
	outlet_blob.release = outlet_release;
	outlet_blob.compare = 0; // rs_compare;
	outlet_blob.write   = 0; // rs_write;

	calibrate_timer(1);
	{
		pthread_t tid;
		pthread_create(&tid, NULL, install_sleep_handler, NULL);
		pthread_detach(tid);
	}
}

	

	
// throws a Prolog exception to signal type error
static int type_error(term_t actual, const char *expected)
{ 
	term_t ex = PL_new_term_ref();

  PL_unify_term(ex, PL_FUNCTOR_CHARS, "error", 2,
		      PL_FUNCTOR_CHARS, "type_error", 2,
		        PL_CHARS, expected,
		        PL_TERM, actual,
		      PL_VARIABLE);

  return PL_raise_exception(ex);
}

/*
static int midi_error(int errno, const char *errmsg, const char *msg)
{ 
	term_t ex = PL_new_term_ref();

  PL_unify_term(ex, PL_FUNCTOR_CHARS, "error", 2,
		      PL_FUNCTOR_CHARS, "midi_send_error", 3,
		        PL_INTEGER, errno,
		        PL_CHARS, errmsg,
		        PL_CHARS, msg,
		      PL_VARIABLE);

  return PL_raise_exception(ex);
}
*/

// put midi_outlet_t data in a Prolog BLOB 
static int unify_outlet(term_t outlet,midi_outlet_t *p) {
	return PL_unify_blob(outlet, p, sizeof(midi_outlet_t), &outlet_blob); 
}

// get midi_outlet_t data from a Prolog BLOB
static int get_outlet(term_t outlet, midi_outlet_t *p)
{ 
	PL_blob_t *type;
	size_t    len;
	midi_outlet_t *p1;
  
	PL_get_blob(outlet, (void **)&p1, &len, &type);
	if (type != &outlet_blob) {
		return type_error(outlet, "plmidi_outlet");
	} else {
		*p=*p1;
		return TRUE;
	}
} 

// get Prolog (Unix) time from a term (should be floating point number)
static int get_prolog_time(term_t time, MIDITimeStamp *ts)
{
	double t;

	if (PL_get_float(time, &t)) {
		*ts = (uint64_t)((t-timebase_origin)*timebase_freq); 
		return TRUE;
	} else {
		return type_error(time,"float");
	}
}

// get an unsigned byte from a numeric atom
static int get_byte(term_t msg, unsigned char *m)
{
	int x;
	if (!PL_get_integer(msg,&x) || x<0 || x>255) return type_error(msg,"uint8");
	*m = x;
	return TRUE;
}


// ------- Foreign interface predicates
//
foreign_t calibrate() { calibrate_timer(1); return TRUE; }

foreign_t listall() { list_devices(); return TRUE; }

foreign_t mk_outlet(term_t id, term_t outlet) 
{ 
	int x;

	if (PL_get_integer(id, &x)) {
		midi_outlet_t o;
		printf("going to open midi destination %d\n",x-1);
		return open_outlet(x-1,&o) && unify_outlet(outlet,&o);
	} else {
		return type_error(id,"integer");
	}
}

foreign_t is_outlet(term_t outlet) 
{ 
	PL_blob_t *type;
	return PL_is_blob(outlet,&type) && type==&outlet_blob;
}


foreign_t send_at(term_t outlet, term_t msg, term_t arg1, term_t arg2, term_t time) 
{
	midi_outlet_t 	o;
	MIDITimeStamp  ts;
	unsigned char	m, a1, a2;

	return get_outlet(outlet,&o) &&
			get_prolog_time(time,&ts) &&
			get_byte(msg, &m) &&
			get_byte(arg1, &a1) &&
			get_byte(arg2, &a2) &&
			send_msg(&o,ts,m,a1,a2);
}


foreign_t send_now(term_t outlet, term_t msg, term_t arg1, term_t arg2)
{
	midi_outlet_t o;
	unsigned char	m, a1, a2;

	return get_outlet(outlet,&o) &&
			get_byte(msg, &m) &&
			get_byte(arg1, &a1) &&
			get_byte(arg2, &a2) &&
			send_msg(&o,0,m,a1,a2);
}



